<?php
/**
 * Check Attendance Status API
 * Returns current attendance status for an employee
 */

header('Content-Type: application/json');
require_once __DIR__ . '/../config/db.php';

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit();
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

// Validate input
if (!isset($input['token'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Token is required']);
    exit();
}

$token = sanitize($input['token']);

try {
    $pdo = getDB();
    
    // Get employee by token
    $stmt = $pdo->prepare("SELECT id FROM employees WHERE qr_token = ?");
    $stmt->execute([$token]);
    $employee = $stmt->fetch();
    
    if (!$employee) {
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'Invalid token']);
        exit();
    }
    
    $employeeId = $employee['id'];
    $today = date('Y-m-d');
    
    // Check existing attendance for today
    $stmt = $pdo->prepare("SELECT check_in_time, check_out_time FROM attendance WHERE employee_id = ? AND date = ?");
    $stmt->execute([$employeeId, $today]);
    $attendance = $stmt->fetch();
    
    $hasCheckedIn = $attendance && $attendance['check_in_time'] ? true : false;
    $hasCheckedOut = $attendance && $attendance['check_out_time'] ? true : false;
    
    echo json_encode([
        'success' => true,
        'hasCheckedIn' => $hasCheckedIn,
        'hasCheckedOut' => $hasCheckedOut,
        'checkInTime' => $hasCheckedIn ? $attendance['check_in_time'] : null,
        'checkOutTime' => $hasCheckedOut ? $attendance['check_out_time'] : null
    ]);
    
} catch (PDOException $e) {
    error_log("Check status error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Server error. Please try again.']);
}
